import pdb

from yamutec.readdata import ReadData
from .core import pyVersion
from . import core
import math

from .utils.bitutils import setallbits

class Device(object):

	def __init__(self, hid_device):
		"""
		:param hid_device: 
		:type hid_device: hidapi.HIDDevice

		"""
		self._hid_device = hid_device

		self._address = -1

		self._path = hid_device.path

		self._is_timeout_active = False

	def _initialize(self):
		self._hid_device.open()

		serial_number = self._hid_device.serial_number

		model = serial_number[-2:]

		self._output_bits = int(core._models[model]['outputs'])
		self._input_bits = int(core._models[model]['inputs'])

		self._output_bytes = int(math.ceil(float(self._output_bits) / 8))
		self._input_bytes = int(math.ceil(float(self._input_bits) / 8))
		
		data = self._hid_device.read(16, 100)

		if data and len(data) == 16:
			if pyVersion == 2:
				self._address = ord(data[8])
			else:
				self._address = data[8]

	@property
	def name(self):
		return self._hid_device.product_string

	@property
	def serial_number(self):
		return self._hid_device.serial_number

	@property
	def address(self):
		return self._address

	@property
	def output_bits(self):
		return self._output_bits

	@property
	def output_bytes(self):
		return self._output_bytes

	@property
	def is_timeout_active(self):
		return self._is_timeout_active

	def _read_raw(self):
		data = self._hid_device.read(16)

		if data and len(data) == 16:
			return data
		else:
			return None

	def read(self):
		"""
		:rtype: ReadData
		"""
		raw_data = self._read_raw()

		if raw_data:
			read_data = ReadData(raw_data, self._output_bits, self._input_bits)
			self._is_timeout_active = read_data.is_timeout_active
			return read_data
		else:
			return None

	def write_raw(self, barray):
		if pyVersion == 2:
			self._hid_device.write(str(barray))
		else:
			self._hid_device.write(bytes(barray))

	def change_outputs(self, wert1, wert2=0, wert3=0, wert4=0, wert5=0, wert6=0, wert7=0, wert8=0):
		barray = bytearray(16)
		barray[0:3] = [0, self._address, 2]
		barray[3:11] = [wert1, wert2, wert3, wert4, wert5, wert6, wert7, wert8]
		self.write_raw(barray)

	def change_output_bytes(self, bytes):
		barray = bytearray(16)
		
		barray[0:3] = [0, self._address, 2]
		barray[3:11] = bytes
		self.write_raw(barray)

	def change_timeout(self, enable, to_value=5, ch1=0, ch2=0, ch3=0, ch4=0, ch5=0, ch6=0, ch7=0, ch8=0):
		"""
		if enable is True, timeout will be activated.  If False it will be deactivated
		If enable is True, the other parameters are required
		"""

		barray = bytearray(16)
		barray[0:3] = [0, self._address, 1]

		if enable:
			barray[3:15] = [1, 1, to_value, 1, ch1, ch2, ch3, ch4, ch5, ch6, ch7, ch8]

		self.write_raw(barray)
		self._is_timeout_active = enable

	def ping(self):
		barray = bytearray(16)
		barray[0:5] = [0, self._address, 1, 1, 2]
		self.write_raw(barray)

	def reboot(self):
		barray = bytearray(16)
		barray[0:4] = [0, self._address, 1, 1]
		self.write_raw(barray)

	def activate_outputs(self, outputs):
		raw_data = self._read_raw()

		if raw_data:
			output_bytes = bytearray(raw_data)
			for output in outputs:
				if output <= self._output_bits:
					bit = output - 1
					index = int(bit / 8)
					output_bytes[index] |= (1 << bit - (index * 8))

			self.change_outputs(
			    output_bytes[0],
			    output_bytes[1],
			    output_bytes[2],
			    output_bytes[3],
			    output_bytes[4],
			    output_bytes[5],
			    output_bytes[6],
			    output_bytes[7],
			)

	def activate_all_outputs(self):
		output_bytes = bytearray(8)

		x = 0

		while x < self._output_bytes:
			if x < self._output_bytes - 1:
				output_bytes[x] = 255
			else:
				remaining_bits = self._output_bits - (x * 8)
				output_bytes[x] = setallbits(0, 1, remaining_bits)

			x += 1

		self.change_output_bytes(output_bytes)

	def deactivate_outputs(self, outputs):
		raw_data = self._read_raw()

		if raw_data:
			output_bytes = bytearray(raw_data)
			for output in outputs:
				if output <= self._output_bits:
					bit = output - 1
					index = int(bit / 8)
					output_bytes[index] &= ~(1 << bit - (index * 8))

			self.change_outputs(
			    output_bytes[0],
			    output_bytes[1],
			    output_bytes[2],
			    output_bytes[3],
			    output_bytes[4],
			    output_bytes[5],
			    output_bytes[6],
			    output_bytes[7],
			)

	def deactivate_all_outputs(self):
		self.change_output_bytes([0, 0, 0, 0, 0, 0, 0, 0])