import threading

from .utils.workers import PeriodicWorker
from .hidapi import LibHID
from .device import Device

from time import sleep


class DeviceServer(object):
	def __init__(self):
		self._stop_evt = threading.Event()

		self.devices_changed_callback = None

		self.read_devices_callback = None

		self.current_device = None #type: Device

		self.read_alldevices_interval = 10 #type: int

		self._devices = {}

		self._devices_list = [] #type: [Device]

		self._check_devices_worker = None

		self._read_devices_worker = None

		self._timeout_worker = None


	def start(self):
		self._check_devices_worker = PeriodicWorker(self._devices_check, self._stop_evt, 500)

		self._timeout_worker = PeriodicWorker(self._timeout_check, self._stop_evt, 200, True)

		self._check_devices_worker.start()
		self._timeout_worker.start()

		if self.read_alldevices_interval > 0:
			self._read_devices_worker = PeriodicWorker(self._read_devices, self._stop_evt, self.read_alldevices_interval)
			self._read_devices_worker.start()



	def stop(self):
		self._stop_evt.set()

		if self._check_devices_worker.is_alive():
			self._check_devices_worker.join()

		if self._timeout_worker.is_alive():
			self._timeout_worker.join()

		if self._read_devices_worker and self._read_devices_worker.is_alive():
			self._read_devices_worker.join()

	@property
	def connected_devices(self):
		"""
		:rtype: [Device]
		"""
		return self._devices_list * 1


	def set_current_device(self, index):
		"""
		:rtype: Device
		"""

		if self._devices_list and index > -1 and index < len(self._devices_list):
			self.current_device = self._devices_list[index]
		else:
			self.current_device = None



		return self.current_device

	def _devices_check(self):
		try:
			current_devices = self._devices

			current_keys = current_devices.keys()

			hid_devices = LibHID.get_devices(vendor_id=0x16d0, product_id=0x0d0e)

			new_devices = {}
			connected = list()
			disconnected = list()

			for hid_device in hid_devices:
				serial_number = hid_device.serial_number
				device = current_devices.get(serial_number)

				if not device:
					device = Device(hid_device)
					device._initialize()
					connected.append(device)

				new_devices[serial_number] = device


			for key in current_keys:
				if key not in new_devices:
					disconnected.append(current_devices[key])

		except Exception as e:
			print('Exception occurred(_devices_check): ' + str(e))


		if len(connected) > 0 or len(disconnected) > 0:
			self._devices = new_devices
			self._devices_list = sorted(new_devices.values(), key=Device.name.fget)

			if self.devices_changed_callback:
				self.devices_changed_callback(connected, disconnected, self._devices_list * 1)

	def _read_devices(self):
		if self.read_devices_callback and self._devices:
			devices = self._devices.values()
			read_datas = []

			for device in devices:
				read_data = device.read()

				if (read_data):
					read_datas.append((device, read_data))

			for t in read_datas:
				self.read_devices_callback(t[0], t[1])


	def _timeout_check(self):
		try:
			for device in self._devices_list:
				if device._is_timeout_active:
					device.ping()
		except Exception as e:
			pass